/* Copyright (C) 1993,1994 by the author(s).
 
 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with ShapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * AtFS -- Attribute Filesystem 
 *
 * cacheadm.c -- administrate derived object cache
 *
 * Author: Andreas Lampen (Andreas.Lampen@cs.tu-berlin.de)
 *
 * $Header: cacheadm.c[3.1] Thu Aug  4 16:03:05 1994 andy@cs.tu-berlin.de frozen $
 */

#include "atfs.h"

int quietMode = FALSE;
int maxCacheAttr = -1;
int maxCacheName = -1;
int maxCacheTotal = -1;

LOCAL int setCacheSize (dir)
     char *dir;
{
  int oldMaxCacheAttr, oldMaxCacheName, oldMaxCacheTotal;

  if (!quietMode && strcmp (dir, "."))
    fprintf (stdout, "%s:\n", dir);
  if (af_cachesize (dir, maxCacheTotal, maxCacheName, maxCacheAttr,
		    &oldMaxCacheTotal, &oldMaxCacheName, &oldMaxCacheAttr) == -1) {
    if (!quietMode)
      fprintf (stderr, "  Error -- %s.\n", af_errmsg ("cachesize"));
    return (1);
  }

  if (!quietMode) {
    /* give feedback */
    if (maxCacheTotal > 0)
      fprintf (stdout, "   total max files: %4d (previously %4d)\n", maxCacheTotal, oldMaxCacheTotal);
    else
      fprintf (stdout, "   total max files: %4d\n", oldMaxCacheTotal);

    if (maxCacheName > 0)
      fprintf (stdout, "max files per name: %4d (previously %4d)\n", maxCacheName, oldMaxCacheName);
    else
      fprintf (stdout, "max files per name: %4d\n", oldMaxCacheName);

    if (maxCacheAttr > 0)
      fprintf (stdout, "max files per attr: %4d (previously %4d)\n", maxCacheAttr, oldMaxCacheAttr);
    else
      fprintf (stdout, "max files per attr: %4d\n", oldMaxCacheAttr);
  }
  return (0);
}

/*==========================================================================
 * main and friends
 *==========================================================================*/

LOCAL void usage ()
{
  fprintf (stderr, "Usage:\tcacheadm [-A max_cached_per_uniq_attr] [-N max_cached_per_name]\n");
  fprintf (stderr, "\t         [-T max_total_cache_size] [-q] [-v] [dirs]\n");
}

extern char *optarg;
extern int  optind;
extern int  errno;
extern char *sys_errlist[];

EXPORT int main (argc, argv)
     int  argc;
     char **argv;
{
  short c, i, ndirs, retCode = 0;

  while ((c = getopt (argc, argv, "A:N:T:qv")) != EOF) {
    switch (c) {
    case 'A': /* max number of versions to be cached per attribute */
      if ((maxCacheAttr = atoi (optarg)) < 0) {
	fprintf (stderr, "  Error -- invalid argument for -A option (must not be negative).");
	exit (1);
      }
      break;
    case 'N': /* max number of versions to be cached per name */
      if ((maxCacheName = atoi (optarg)) < 0) {
	fprintf (stderr, "  Error -- invalid argument for -N option (must not be negative).");
	exit (1); 
      }
      break;
    case 'q': /* quiet mode  */
      quietMode = TRUE;
      break;
    case 'T': /* total max number of versions to be cached */
      if ((maxCacheTotal = atoi (optarg)) < 0) {
	fprintf (stderr, "  Error -- invalid argument for -T option (must not be negative).");
	exit (1);
      }
      break;
    case 'v': /* print current version of this program */
      printf ("This is cacheadm version %s.\n", af_version());
      exit (0);
    default:
      usage ();
      exit (1);
    }
  }  /* end of command line parsing */

  ndirs = argc - optind;
  if (ndirs == 0) {
    retCode += setCacheSize (".");
  }
  else {
    /* process arguments */
    for (i = 0; i < ndirs; i++) {
      struct stat iBuf;
      /* check if argument exists and is a directory */
      if ((stat (argv[i+optind], &iBuf) != ERROR) && S_ISDIR(iBuf.st_mode)) {
	retCode += setCacheSize (argv[i+optind]);
      }
      else {
	fprintf (stderr, "  Error -- %s: %s\n", argv[i+optind], sys_errlist[errno]);
      }
    }
  }
  exit (retCode);
} /* end of main */
