//driver.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2019
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include "driver.h"
#include "autodetected.h"

#ifdef ROAR_HAVE_AD_ESDFW
#include "driver_esdfw.h"
#endif

static int roar_cdriver_null(struct roar_vio_calls * calls, const char * name, const char * dev, struct roar_audio_info * info, int dir, struct roar_keyval * para, ssize_t paralen) {
 ROAR_DBG("roar_cdriver_null(calls=%p, name='%s', dev='%s', info=%p{...}, dir=%i(?)) = ?", calls, name, dev, info, dir);

 (void)name, (void)dev, (void)info, (void)dir, (void)para, (void)paralen;

 memset(calls, 0, sizeof(struct roar_vio_calls));
 calls->flags = ROAR_VIO_FLAGS_NONE;
 calls->refc  = 1;
 calls->read  = roar_vio_null_rw;
 calls->write = roar_vio_null_rw;

 ROAR_DBG("roar_cdriver_null(calls=%p, name='%s', dev='%s', info=%p{...}, dir=%i(?)) = 0", calls, name, dev, info, dir);
 return 0;
}

struct {
 const char * name;
 int  (*open)(struct roar_vio_calls * calls, const char * name, const char * dev, struct roar_audio_info * info, int dir);
 int  (*open2)(struct roar_vio_calls * calls, const char * name, const char * dev, struct roar_audio_info * info, int dir, struct roar_keyval * para, ssize_t paralen);
} _g_roar_cdriver[] = {
 {"null", NULL, roar_cdriver_null},
 {"roar", roar_cdriver_roar, NULL},
#if defined(ROAR_HAVE_OSS_BSD) || defined(ROAR_HAVE_OSS)
 {"oss",  roar_cdriver_oss, NULL},
#endif
#ifdef ROAR_HAVE_AD_ESDFW
#include "driver_esdfw.c"
#endif
 {NULL, NULL, NULL}
};


static int roar_cdriver_open_default(struct roar_vio_calls * calls, const char * dev, struct roar_audio_info * info, int dir, struct roar_keyval * para, ssize_t paralen) {
 const char * names[] = {
  // native APIs:
  "oss",
  // Virtual APIs:
  "roar",
  // Dummys:
  "null"
 };
 size_t i;
 int ret;

 if ( dev != NULL ) {
  ROAR_WARN("roar_cdriver_open_default(calls=%p, dev='%s', info=%p, dir=%i): Try to open given device without driver name is a bad thing to do. This is normally a Application error.", calls, dev, info, dir);
 }

 for (i = 0; i < (sizeof(names)/sizeof(*names)); i++) {
  ret = roar_cdriver_open2(calls, names[i], dev, info, dir, para, paralen);
  if ( ret != -1 )
   return ret;
 }

 // roar_error is still set to the error of the last tested driver.
 return -1;
}

int roar_cdriver_open2(struct roar_vio_calls * calls, const char * name, const char * dev, struct roar_audio_info * info, int dir, struct roar_keyval * para, ssize_t paralen) {
 int i;
 char *delm;

 if ( name == NULL )
  return roar_cdriver_open_default(calls, dev, info, dir, para, paralen);

 for (i = 0; _g_roar_cdriver[i].name != NULL; i++) {
  ROAR_DBG("roar_cdriver_open(*): _g_roar_cdriver[i].name='%s' <cmp> name='%s'", _g_roar_cdriver[i].name, name);

  if ( !strcmp(_g_roar_cdriver[i].name, name) ) {
   if ( _g_roar_cdriver[i].open2 != NULL )
    return _g_roar_cdriver[i].open2(calls, name, dev, info, dir, para, paralen);
   if ( _g_roar_cdriver[i].open != NULL )
    return _g_roar_cdriver[i].open(calls, name, dev, info, dir);
  }

  if ( (delm = strstr(_g_roar_cdriver[i].name, ":")) != NULL ) {
   ROAR_DBG("roar_cdriver_open(*): delm+1='%s' <cmp> name='%s'", delm+1, name);
   if ( !strcmp(delm+1, name) ) {
    if ( _g_roar_cdriver[i].open2 != NULL )
     return _g_roar_cdriver[i].open2(calls, name, dev, info, dir, para, paralen);
    if ( _g_roar_cdriver[i].open != NULL )
     return _g_roar_cdriver[i].open(calls, name, dev, info, dir);
   }
  }

 }

 roar_err_set(ROAR_ERROR_NOENT);
 return -1;
}

ssize_t roar_cdriver_list   (const char ** list, size_t len, size_t offset) {
 size_t i;
 size_t idx = 0;

 if ( list == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 if ( len == 0 )
  return 0;

 if ( offset >= (sizeof(_g_roar_cdriver)/sizeof(*_g_roar_cdriver)) )
  return 0;

 for (i = offset; _g_roar_cdriver[i].name != NULL; i++) {
  list[idx++] = _g_roar_cdriver[i].name;
 }

 return idx;
}

int roar_cdriver_open(struct roar_vio_calls * calls, const char * name, const char * dev, struct roar_audio_info * info, int dir) {
 return roar_cdriver_open2(calls, name, dev, info, dir, NULL, -1); 
}

//ll
