% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_spectrum_region.R
\name{plot_spectrum_region}
\alias{plot_spectrum_region}
\title{Plot point mutation spectrum per genomic region}
\usage{
plot_spectrum_region(
  type_occurrences,
  by = NA,
  mode = c("relative_sample_feature", "relative_sample", "absolute"),
  indv_points = FALSE,
  error_bars = c("95\%_CI", "stdev", "SEM", "none"),
  colors = NULL,
  legend = TRUE,
  condensed = FALSE
)
}
\arguments{
\item{type_occurrences}{Type occurrences matrix}

\item{by}{Optional grouping variable}

\item{mode}{The y-axis plotting mode.
* 'relative_sample', the number of variants will be shown
divided by the total number of variants in that sample;
* 'relative_sample_feature', the number of variants will be shown
divided by the total number of variants in that sample and genomic region (Default);
* 'absolute' The absolute number of mutations is shown;}

\item{indv_points}{Whether to plot the individual samples
as points, default = FALSE}

\item{error_bars}{The type of error bars to plot.
* '95%_CI' for 95% Confidence intervals (Default);
* 'stdev' for standard deviations;
* 'SEM' for the standard error of the mean (NOT recommended);
* 'none' Do not plot any error bars;}

\item{colors}{Optional color vector with 7 values}

\item{legend}{Plot legend, default = TRUE}

\item{condensed}{More condensed plotting format. Default = F.}
}
\value{
Spectrum plot by genomic region
}
\description{
A spectrum similar to the one from 'plot_spectrum()' is plotted.
However the spectrum is plotted separately per genomic region.
As input it takes a 'type_occurrences' matrix that was calculated per genomic region.
To get a 'type_occurrences' matrix per region,
first use the 'split_muts_region()' function on a GR or GRangesList.
Then use the 'mut_type_occurrences' as you would normally.
The by, colors and legend argument work the same as in 'plot_spectrum()'.
}
\details{
The y-axis can be plotted with three different modes. With
'relative_sample_feature', the number of variants will be shown divided by
the total number of variants in that sample and genomic region. This is
generally the most usefull, because it allows you to compare the spectra off
different regions. When you use 'relative_sample', the number of variants
will be shown divided by the total number of variants in that sample. This
can be usefull when you want to compare the number of mutations between
regions. Finally, when you use 'absolute', the absolute mutation numbers are
shown. This can be usefull when you want to compare the mutation load between
different groups of samples.
}
\examples{
## See the 'split_muts_region()' example for how we obtained the
## following data:
grl <- readRDS(system.file("states/grl_split_region.rds",
  package = "MutationalPatterns"
))


## Load a reference genome.
ref_genome <- "BSgenome.Hsapiens.UCSC.hg19"
library(ref_genome, character.only = TRUE)


## Get the type occurrences for all VCF objects.
type_occurrences <- mut_type_occurrences(grl, ref_genome)

## Plot the relative point mutation spectrum per genomic region
plot_spectrum_region(type_occurrences)

## Include the individual sample points
plot_spectrum_region(type_occurrences, indv_points = TRUE)

## Plot the relative point mutation spectrum per genomic region,
## but normalize only for the samples
plot_spectrum_region(type_occurrences, mode = "relative_sample")

## Plot the absolute point mutation spectrum per genomic region
plot_spectrum_region(type_occurrences, mode = "absolute")

## Plot the point mutations spectrum with different error bars
plot_spectrum_region(type_occurrences, error_bars = "stdev")

## Plot the relative point mutation spectrum per sample type and per genomic region
## Determine tissue names
tissue <- c(
  "colon", "colon", "colon",
  "intestine", "intestine", "intestine",
  "liver", "liver", "liver"
)
plot_spectrum_region(type_occurrences, by = tissue)

## Plot the relative point mutation spectrum per individual sample and per genomic region
## Determine sample names
sample_names <- c(
  "colon1", "colon2", "colon3",
  "intestine1", "intestine2", "intestine3",
  "liver1", "liver2", "liver3"
)

plot_spectrum_region(type_occurrences, by = sample_names, error_bars = "none")

## Plot it in a more condensed manner, 
## which is is ideal for publications.
plot_spectrum_region(type_occurrences, 
by = sample_names, 
error_bars = "none", 
condensed = TRUE)

}
\seealso{
\code{\link{read_vcfs_as_granges}},
\code{\link{mut_type_occurrences}},
\code{\link{plot_spectrum}},
\code{\link{split_muts_region}}

Other genomic_regions: 
\code{\link{bin_mutation_density}()},
\code{\link{lengthen_mut_matrix}()},
\code{\link{plot_profile_region}()},
\code{\link{split_muts_region}()}
}
\concept{genomic_regions}
