/*
 * Copyright (C) by Argonne National Laboratory
 *     See COPYRIGHT in top-level directory
 */

#include "mpiimpl.h"

/* Remote send local scatter
 *
 * Root sends to rank 0 in remote group.  rank 0 does local
 * intracommunicator scatter (binomial tree).
 *
 * Cost: (lgp+1).alpha + n.((p-1)/p).beta + n.beta
 */

int MPIR_Scatter_inter_remote_send_local_scatter(const void *sendbuf, MPI_Aint sendcount,
                                                 MPI_Datatype sendtype, void *recvbuf,
                                                 MPI_Aint recvcount, MPI_Datatype recvtype,
                                                 int root, MPIR_Comm * comm_ptr, int coll_attr)
{
    int rank, local_size, remote_size, mpi_errno = MPI_SUCCESS;
    MPI_Status status;
    MPIR_Comm *newcomm_ptr = NULL;
    MPIR_CHKLMEM_DECL();

    if (root == MPI_PROC_NULL) {
        /* local processes other than root do nothing */
        return MPI_SUCCESS;
    }

    remote_size = comm_ptr->remote_size;
    local_size = comm_ptr->local_size;

    if (root == MPI_ROOT) {
        /* root sends all data to rank 0 on remote group and returns */
        mpi_errno =
            MPIC_Send(sendbuf, sendcount * remote_size, sendtype, 0, MPIR_SCATTER_TAG, comm_ptr,
                      coll_attr);
        MPIR_ERR_CHECK(mpi_errno);
        goto fn_exit;
    } else {
        /* remote group. rank 0 receives data from root. need to
         * allocate temporary buffer to store this data. */
        MPI_Aint recvtype_sz;
        void *tmp_buf = NULL;

        rank = comm_ptr->rank;

        if (rank == 0) {
            MPIR_Datatype_get_size_macro(recvtype, recvtype_sz);
            MPIR_CHKLMEM_MALLOC(tmp_buf, recvcount * local_size * recvtype_sz);

            mpi_errno = MPIC_Recv(tmp_buf, recvcount * local_size * recvtype_sz, MPIR_BYTE_INTERNAL,
                                  root, MPIR_SCATTER_TAG, comm_ptr, &status);
            MPIR_ERR_CHECK(mpi_errno);
        } else {
            /* silence -Wmaybe-uninitialized due to MPIR_Scatter by non-zero ranks */
            recvtype_sz = 0;
        }

        /* Get the local intracommunicator */
        if (!comm_ptr->local_comm)
            MPII_Setup_intercomm_localcomm(comm_ptr);

        newcomm_ptr = comm_ptr->local_comm;

        /* now do the usual scatter on this intracommunicator */
        mpi_errno = MPIR_Scatter(tmp_buf, recvcount * recvtype_sz, MPIR_BYTE_INTERNAL,
                                 recvbuf, recvcount, recvtype, 0, newcomm_ptr, coll_attr);
        MPIR_ERR_CHECK(mpi_errno);
    }

  fn_exit:
    MPIR_CHKLMEM_FREEALL();
    return mpi_errno;
  fn_fail:
    goto fn_exit;
}
