/*
 *  $Id: mask_distribute.c 28821 2025-11-06 16:00:41Z yeti-dn $
 *  Copyright (C) 2003-2021 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef enum {
    DISTRIBUTE_WITHIN_FILE  = 0,
    DISTRIBUTE_TO_ALL_FILES = 1,
} MaskDistribMode;

enum {
    PARAM_KEEP_EXISTING,
    PARAM_MODE,
};

typedef struct {
    GwyParams *params;
    GwyField *mask;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GArray *undo_quarks;
    GwyFile *file;
    gint id;
} MaskDistribData;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static GwyDialogOutcome run_gui             (ModuleArgs *args);
static void             distribute_in_one   (GwyFile *file,
                                             gpointer user_data);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Distributes masks to other channels."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2015",
};

GWY_MODULE_QUERY2(module_info, mask_distribute)

static gboolean
module_register(void)
{
    gwy_process_func_register("mask_distribute",
                              module_main,
                              N_("/_Mask/_Distribute..."),
                              GWY_ICON_MASK_DISTRIBUTE,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Distribute mask to other channels"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static const GwyEnum modes[] = {
        { N_("Channels within the file"), DISTRIBUTE_WITHIN_FILE,  },
        { N_("Channels in all files"),    DISTRIBUTE_TO_ALL_FILES, },
    };
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_process_func_current());
    gwy_param_def_add_boolean(paramdef, PARAM_KEEP_EXISTING, "keep_existing", _("Preserve existing masks"), FALSE);
    gwy_param_def_add_gwyenum(paramdef, PARAM_MODE, "mode", _("Distribute to"),
                              modes, G_N_ELEMENTS(modes), DISTRIBUTE_WITHIN_FILE);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    MaskDistribData distdata;
    ModuleArgs args;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_MASK_FIELD, &args.mask,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(args.mask && id >= 0);

    args.params = gwy_params_new_from_settings(define_module_params());
    if (mode !=GWY_RUN_IMMEDIATE) {
        GwyDialogOutcome outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome != GWY_DIALOG_PROCEED)
            goto end;
    }

    distdata.file = data;
    distdata.id = id;
    distdata.args = &args;
    distdata.undo_quarks = g_array_new(FALSE, FALSE, sizeof(GQuark));

    if (gwy_params_get_enum(args.params, PARAM_MODE) == DISTRIBUTE_TO_ALL_FILES)
        gwy_data_browser_foreach(distribute_in_one, &distdata, NULL);
    else
        distribute_in_one(data, &distdata);

    g_array_free(distdata.undo_quarks, TRUE);

end:
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    GwyDialog *dialog;
    GwyParamTable *table;
    ModuleGUI gui;

    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Distribute Mask"));
    dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GWY_RESPONSE_RESET, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_radio(table, PARAM_MODE);
    gwy_param_table_append_separator(table);
    gwy_param_table_append_checkbox(table, PARAM_KEEP_EXISTING);

    gwy_dialog_add_content(dialog, gwy_param_table_widget(gui.table), FALSE, TRUE, 0);
    gwy_dialog_add_param_table(dialog, gui.table);

    return gwy_dialog_run(dialog);
}

static void
distribute_in_one(GwyFile *file, gpointer user_data)
{
    MaskDistribData *distdata = (MaskDistribData*)user_data;
    GwyField *field, *mask;
    GwyDataMismatchFlags flags = GWY_DATA_MISMATCH_RES | GWY_DATA_MISMATCH_REAL;
    gboolean keep_existing = gwy_params_get_boolean(distdata->args->params, PARAM_KEEP_EXISTING);
    GArray *undo_quarks = distdata->undo_quarks;
    GQuark quark;
    gint *ids;
    guint i;

    g_array_set_size(undo_quarks, 0);
    ids = gwy_file_get_ids(file, GWY_FILE_IMAGE);

    for (i = 0; ids[i] >= 0; i++) {
        if (file == distdata->file && ids[i] == distdata->id)
            continue;

        field = gwy_file_get_image(file, ids[i]);
        g_return_if_fail(field);
        mask = gwy_file_get_image_mask(file, ids[i]);

        if (mask && keep_existing)
            continue;
        if (gwy_field_is_incompatible(distdata->args->mask, field, flags))
            continue;

        g_array_append_val(undo_quarks, quark);
        gwy_log_add(file, GWY_FILE_IMAGE, -1, ids[i]);
    }
    g_free(ids);

    if (!undo_quarks->len)
        return;

    gwy_app_undo_qcheckpointv(GWY_CONTAINER(file), undo_quarks->len, (GQuark*)undo_quarks->data);
    for (i = 0; i < undo_quarks->len; i++) {
        quark = g_array_index(undo_quarks, GQuark, i);
        mask = gwy_field_copy(distdata->args->mask);
        gwy_container_pass_object(GWY_CONTAINER(file), quark, mask);
    }
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
