// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package path

import (
	"fmt"
	"strings"

	"code.forgejo.org/f3/gof3/v3/id"
)

type Implementation []PathElement

func (o Implementation) PathString() PathString {
	elements := NewPathString()
	for i, e := range o {
		eid := e.GetID()
		// i == 0 is root and intentionally empty
		if i > 0 && eid == id.NilID {
			eid = id.NewNodeID("nothing")
		}
		elements.Append(eid.String())
	}
	return elements
}

func (o Implementation) PathMappedString() PathString {
	elements := NewPathString()
	for _, e := range o {
		elements.Append(e.GetMappedID().String())
	}
	return elements
}

func (o Implementation) String() string {
	return o.PathString().Join()
}

var replacer = strings.NewReplacer(
	"{", "%7B",
	"}", "%7D",
)

func (o Implementation) ReadablePathString() PathString {
	elements := NewPathString()
	if o.Length() > 0 {
		elements.Append("")
		for _, e := range o[1:] {
			element := e.GetID().String()
			if f := e.ToFormat(); f != nil {
				name := f.GetName()
				if element != name {
					element = fmt.Sprintf("{%s/%s}", replacer.Replace(name), element)
				}
			}
			elements.Append(element)
		}
	}
	return elements
}

func (o Implementation) ReadableString() string {
	return o.ReadablePathString().Join()
}

func (o Implementation) Length() int {
	return len(o)
}

func (o Implementation) Append(child PathElement) Path {
	return append(o, child)
}

func (o Implementation) PopFirst() (PathElement, Path) {
	return o.First(), o.RemoveFirst()
}

func (o Implementation) RemoveFirst() Path {
	return o[1:]
}

func (o Implementation) Pop() (PathElement, Path) {
	return o.Last(), o.RemoveLast()
}

func (o Implementation) RemoveLast() Path {
	return o[:len(o)-1]
}

func (o Implementation) Empty() bool {
	return len(o) == 0
}

func (o Implementation) Last() PathElement {
	return o[len(o)-1]
}

func (o Implementation) First() PathElement {
	return o[0]
}

func (o Implementation) All() []PathElement {
	return o
}
