import pytest
import sys

from pathlib import Path
from fpdf import FPDF
from test.conftest import assert_pdf_equal, check_signature, EPOCH

HERE = Path(__file__).resolve().parent
TRUSTED_CERT_PEMS = (HERE / "signing-certificate.crt",)


def test_sign_pkcs12(tmp_path):
    pdf = FPDF()
    pdf.set_creation_date(EPOCH)
    pdf.add_page()
    pdf.sign_pkcs12(HERE / "signing-certificate.p12", password=b"fpdf2")
    assert_pdf_equal(pdf, HERE / "sign_pkcs12.pdf", tmp_path)
    check_signature(pdf, TRUSTED_CERT_PEMS)


@pytest.mark.skipif(
    sys.platform in ("cygwin", "win32") and sys.version_info[:2] == (3, 14),
    reason="Skipped on Windows with Python 3.14 due to zlib compressed data differences",
)
def test_sign_pkcs12_with_link(tmp_path):
    "This test ensures that Signature & Link annotations can be combined"
    pdf = FPDF()
    pdf.set_creation_date(EPOCH)
    pdf.set_font("Helvetica", size=30)
    pdf.add_page()
    pdf.text(x=80, y=50, text="Page 1/2")
    pdf.add_page()
    pdf.set_xy(80, 50)
    pdf.sign_pkcs12(HERE / "signing-certificate.p12", password=b"fpdf2")
    pdf.cell(
        w=50,
        h=20,
        text="URL link",
        border=1,
        align="C",
        link="https://github.com/py-pdf/fpdf2",
    )
    assert_pdf_equal(pdf, HERE / "sign_pkcs12_with_link.pdf", tmp_path)
    check_signature(pdf, TRUSTED_CERT_PEMS)
