# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Debusine command line interface, asset management commands."""

import argparse

from debusine.assets import AssetCategory, asset_data_model
from debusine.client.commands.base import (
    DebusineCommand,
    RequiredInputDataCommand,
    WorkspaceCommand,
)
from debusine.client.models import model_to_json_serializable_list


class Create(RequiredInputDataCommand, WorkspaceCommand, group="asset"):
    """Create an asset."""

    @classmethod
    def configure(cls, parser: argparse.ArgumentParser) -> None:
        """Configure the ArgumentParser for this subcommand."""
        super().configure(parser)
        parser.add_argument(
            "category",
            type=AssetCategory,
            help="Category of asset",
        )

    def run(self) -> None:
        """Run the command."""
        category = self.args.category

        asset_data = asset_data_model(category, self.input_data)

        with self._api_call_or_fail():
            asset = self.debusine.asset_create(
                category=category,
                data=asset_data,
                workspace=self.workspace,
            )

        # TODO: Ideally this would show a URL instead like everything
        # else, but assets don't have a web view at the moment.
        self.print_yaml_output(
            {
                "result": "success",
                "message": (
                    f"New asset created in {self.debusine.base_api_url} "
                    f"in workspace {asset.workspace} with id {asset.id}."
                ),
            }
        )


class LegacyCreate(
    Create, name="create-asset", deprecated="see `asset create`"
):
    """Create an asset."""


class List(DebusineCommand, group="asset"):
    """List assets."""

    @classmethod
    def configure(cls, parser: argparse.ArgumentParser) -> None:
        """Configure the ArgumentParser for this subcommand."""
        super().configure(parser)
        parser.add_argument(
            "--id",
            metavar="ID",
            type=int,
            help="Show an individual asset by ID.",
        )
        parser.add_argument(
            "--workspace",
            "-w",
            metavar="NAME",
            type=str,
            help="List all assets in workspace.",
        )
        parser.add_argument(
            "--work-request",
            metavar="ID",
            type=int,
            help="List assets created by work request.",
        )

    def run(self) -> None:
        """Run the command."""
        asset = self.args.id
        workspace = self.args.workspace
        work_request = self.args.work_request
        if asset is None and workspace is None and work_request is None:
            self._fail(
                "At least one of --id, --workspace, and --work-request must "
                "be specified."
            )
        with self._api_call_or_fail():
            assets = self.debusine.asset_list(
                asset_id=asset,
                work_request=work_request,
                workspace=workspace,
            )

        self.print_yaml_output(model_to_json_serializable_list(assets))


class LegacyList(List, name="list-assets", deprecated="see `asset list`"):
    """List assets."""
